using System;
using System.Collections;
using System.Data;
using System.Data.SqlClient;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.DAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Cameron Taylor</Developer>
	///	<Developer>Greg Lohse</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>8/23/2002</CreationDate>
	///<Note>Note: The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///	Data access layer class implementing database read/write functionality for local supplier related data.
	///</summary>

	#endregion

	public class LocalSupplier 
	{
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7540"> 
		///		<ExpectedInput>Facility Guid</ExpectedInput>
		///		<ExpectedOutput>DataRow</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7541"> 
		///		<ExpectedInput>Invalid Facility Guid</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves facility (local supplier) data by facility GUID (not local supplier guid).
		/// </summary>
		/// <param name="facilityGuid">Facility GUID.</param>
		/// <returns><see cref="DataRow"/> containing data</returns>
		public static DataRow GetFacilityDefinitionByGuid( Guid facilityGuid )
		{
			SqlParameter[] _prms =
			{
				new SqlParameter( STOREDPROC.GetFacilityByFacilityGuid.facilityguid, System.Data.SqlDbType.UniqueIdentifier ),
			};
			
			_prms[0].Value = facilityGuid;

			return Common.StoredProcedure.GetSingleDataRow( STOREDPROC.GetFacilityByFacilityGuid.StoredProcName, _prms );
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1101"> 
		///		<ExpectedInput>Facility Name, FDARegistration Number</ExpectedInput>
		///		<ExpectedOutput>DataTable of LocalSupplier Details</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1103"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Retrieves a datatable of local suppliers based on input parameters
		/// </summary>
		/// <param name="facilityName">Can be empty or partial</param>
		/// <param name="fdaRegistrationNumber">Can be empty or partial</param>
		/// <returns>DataTable</returns>
		public static DataTable GetLocalSupplierLookupList(string facilityName, string fdaRegistrationNumber)
		{
			SqlParameter[] _prms = 
				{
					new SqlParameter(STOREDPROC.GetLocalSupplierLookupList.facilityname, SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetLocalSupplierLookupList.fdaregistrationnumber, SqlDbType.Char,9),
					new SqlParameter(STOREDPROC.GetLocalSupplierLookupList.divisioncode, SqlDbType.Char)
				};

			_prms[0].Value = (facilityName.Trim() == String.Empty) ? null : facilityName;
			_prms[1].Value = (fdaRegistrationNumber.Trim() == String.Empty) ? null : fdaRegistrationNumber;
			_prms[2].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetDataTable( STOREDPROC.GetLocalSupplierLookupList.StoredProcName, _prms );
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5024"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Non-empty list of national facilities.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5025"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves a list of national facilities from the DB.
		/// </summary>
		/// <returns><see cref="DataTable"/> containing a list of active national facilities.</returns>
		public static DataTable GetFacilityNationalLookupList()
		{
			return Common.StoredProcedure.GetDataTable( STOREDPROC.GetFacilityNationalLookupList.StoredProcName );
		}
		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1115"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable of LocalSuppliers for current division</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1117"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets a datatable of available local suppliers for the current user's division.
		/// </summary>
		/// <returns>DataTable</returns>
		public static DataTable GetLocalSuppliersForDivision()
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetLocalSuppliersForDivision.divisioncode, System.Data.SqlDbType.VarChar) 
			};
			
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			
			return Common.StoredProcedure.GetDataTable(STOREDPROC.GetLocalSuppliersForDivision.StoredProcName, prms);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1125"> 
		///		<ExpectedInput>FdaRegistration Number, Boolean (Return Only Active Suppliers)</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1127"> 
		///		<ExpectedInput>Invalid fdaRegistration#</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets the local supplier information for a given FDA registration number.
		/// </summary>
		/// <param name="fdaRegistrationNumber"></param>
		/// <param name="activeOnly">Include inactive local suppliers when quereying data.</param>
		/// <returns>DataTable</returns>
		public static DataTable GetLocalSupplierDetailByFdaRegNum(string fdaRegistrationNumber, bool activeOnly)
		{			
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetLocalSupplierByFdaRegistrationNumber.fdaregistrationnumber, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetLocalSupplierByFdaRegistrationNumber.activeonly, System.Data.SqlDbType.Bit),
				new SqlParameter(STOREDPROC.GetLocalSupplierByFdaRegistrationNumber.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = fdaRegistrationNumber;
			prms[1].Value = activeOnly;
			prms[2].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetDataTable(STOREDPROC.GetLocalSupplierByFdaRegistrationNumber.StoredProcName, prms);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7542"> 
		///		<ExpectedInput>ICCBBA Number</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7543"> 
		///		<ExpectedInput>Invalid ICCBBA Number</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetLocalSupplierDetailByIccbaRegNum
		/// </summary>
		/// <param name="iccbaRegistrationNumber"></param>
		/// <param name="activeOnly"></param>
		/// <returns></returns>
		public static DataTable GetLocalSupplierDetailByIccbaRegNum(string iccbaRegistrationNumber, bool activeOnly)
		{			
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetLocalSupplierByICCBARegistrationNumber.iccbaregistrationnumber, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetLocalSupplierByICCBARegistrationNumber.activeonly, System.Data.SqlDbType.Bit),
				new SqlParameter(STOREDPROC.GetLocalSupplierByICCBARegistrationNumber.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = iccbaRegistrationNumber;
			prms[1].Value = activeOnly;
			prms[2].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetDataTable(STOREDPROC.GetLocalSupplierByICCBARegistrationNumber.StoredProcName, prms);
		}
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1129"> 
		///		<ExpectedInput>LocalSupplierGuid, (Boolean)Return only Active Suppliers</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1130"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets local supplier information for a given localSupplierGUID
		/// </summary>
		/// <param name="localSupplierGuid"></param>
		/// <param name="activeOnly"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetLocalSupplierDetailByGuid(System.Guid localSupplierGuid, bool activeOnly)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetLocalSupplierByGuid.localsupplierguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetLocalSupplierByGuid.activeonly, System.Data.SqlDbType.Bit),
			};
			
			prms[0].Value = localSupplierGuid;
			prms[1].Value = activeOnly;

			return Common.StoredProcedure.GetDataTable(STOREDPROC.GetLocalSupplierByGuid.StoredProcName, prms);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1131"> 
		///		<ExpectedInput>FDA Registration Number</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1136"> 
		///		<ExpectedInput>Invalid Number</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// If the FDAReg# is found for the current user's division, and it's active, this returns true
		/// </summary>
		/// <param name="fdaRegistrationNumber"></param>
		/// <returns>bool</returns>
		public static bool IsValidFdaRegistrationNumber(string fdaRegistrationNumber)
		{			
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.IsValidFdaRegistrationNumber.fdaregistrationnumber, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.IsValidFdaRegistrationNumber.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = fdaRegistrationNumber;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return new Common.StoredProcedure().GetBoolReturnValue( STOREDPROC.IsValidFdaRegistrationNumber.StoredProcName, prms );
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1138"> 
		///		<ExpectedInput>LocalSupplierGuid</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1139"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// If there are any active bloodproducts associated with this local supplier, return true
		/// </summary>
		/// <param name="localSupplierGuid"></param>
		/// <returns>bool</returns>
		public static bool IsLocalSupplierOnActiveBloodProducts(System.Guid localSupplierGuid)
		{			
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.IsLocalSupplierOnActiveBloodProducts.localsupplierguid, System.Data.SqlDbType.UniqueIdentifier)
			};
			
			prms[0].Value = localSupplierGuid;

			return new Common.StoredProcedure().GetBoolReturnValue( STOREDPROC.IsLocalSupplierOnActiveBloodProducts.StoredProcName, prms );
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6628"> 
		///		<ExpectedInput>IccbbaRegistrationNumber</ExpectedInput>
		///		<ExpectedOutput>Boolean</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6629"> 
		///		<ExpectedInput>Invalid Number</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BR_1.64
		/// </summary>
		/// <param name="localSupplierIccbbaNumber"></param>
		/// <returns>bool</returns>
		public static bool IsLocalSupplierActiveByIccbbaRegistrationNumber(string localSupplierIccbbaNumber)
		{			
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.IsLocalSupplierActiveByIccbbaRegNum.iccbbaregistrationnumber, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.IsLocalSupplierActiveByIccbbaRegNum.divisioncode,  System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = localSupplierIccbbaNumber;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return new Common.StoredProcedure().GetBoolReturnValue( STOREDPROC.IsLocalSupplierActiveByIccbbaRegNum.StoredProcName, prms );
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1143"> 
		///		<ExpectedInput>FdaRegistraionNumber</ExpectedInput>
		///		<ExpectedOutput>Boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1144"> 
		///		<ExpectedInput>Invalid Number</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// If there are any active bloodunits associated with this local supplier, return true
		/// </summary>
		/// <param name="fdaRegistrationNumber"></param>
		/// <returns>bool</returns>
		public static bool IsLocalSupplierOnActiveBloodUnits(string fdaRegistrationNumber)
		{			
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.IsLocalSupplierOnActiveBloodUnits.fdaregistrationnumber, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.IsLocalSupplierOnActiveBloodUnits.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = fdaRegistrationNumber;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return new Common.StoredProcedure().GetBoolReturnValue( STOREDPROC.IsLocalSupplierOnActiveBloodUnits.StoredProcName, prms );
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1145"> 
		///		<ExpectedInput>DataTable of Supplier Detials, LastUpdateFunctionID</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1146"> 
		///		<ExpectedInput>Existing Guids</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Saves the current Local Supplier, returns true on success
		/// </summary>
		/// <param name="dtAddress"></param>
		/// <param name="dtLocalSupplier"></param>
		/// <param name="newAddress">Indicates if updating or inserting an address record</param>
		/// <returns>bool</returns>
		public static bool InsertLocalSupplier(DataTable dtLocalSupplier, DataTable dtAddress, bool newAddress)
		{	
			System.Collections.ArrayList dtArray = new ArrayList();
			System.Collections.ArrayList sprocArray = new ArrayList();

			dtArray.Add(dtAddress);
			if (newAddress)
				sprocArray.Add(STOREDPROC.InsertAddress.StoredProcName);
			else
                sprocArray.Add(STOREDPROC.UpdateAddress.StoredProcName);

			dtArray.Add(dtLocalSupplier);
			sprocArray.Add(STOREDPROC.InsertLocalSupplier.StoredProcName);

			int rc = new Common.StoredProcedure().TransactionalGetValue(sprocArray, dtArray);

			return(rc == 0);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1147"> 
		///		<ExpectedInput>DataTable with Supplier Information, Address DataTable</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1149"> 
		///		<ExpectedInput>Incorrect RowVersions</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Saves the current Local Supplier, returns true on success
		/// </summary>
		/// <param name="dtAddress"></param>
		/// <param name="dtLocalSupplier"></param>
		/// <returns>bool</returns>
		public static bool UpdateLocalSupplier(DataTable dtLocalSupplier, DataTable dtAddress)
		{	
			System.Collections.ArrayList dtArray = new ArrayList();
			System.Collections.ArrayList sprocArray = new ArrayList();

			dtArray.Add(dtAddress);
			sprocArray.Add(STOREDPROC.UpdateAddress.StoredProcName);

			dtArray.Add(dtLocalSupplier);
			sprocArray.Add(STOREDPROC.UpdateLocalSupplier.StoredProcName);

			int rc = new Common.StoredProcedure().TransactionalGetValue(sprocArray, dtArray);

			return(rc == 0);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7545"> 
		///		<ExpectedInput>true</ExpectedInput>
		///		<ExpectedOutput>DataTable with RowVersion</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7546"> 
		///		<ExpectedInput>false</ExpectedInput>
		///		<ExpectedOutput>DataTable without RowVersion</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// spInsertLocalSupplier, spUpdateLocalSupplier
		/// </summary>
		/// <param name="isUpdate"></param>
		/// <returns></returns>
		public static System.Data.DataTable GetEmptyLocalSupplierTableSchema(bool isUpdate)
		{
			System.Data.DataTable dtLocalSupplier = new DataTable(TABLE.LocalSupplier.TableName);

			dtLocalSupplier.Columns.Add(TABLE.LocalSupplier.LocalSupplierGuid, typeof(System.Guid));
			dtLocalSupplier.Columns.Add(TABLE.LocalSupplier.FacilityName, typeof(string));
			dtLocalSupplier.Columns.Add(TABLE.LocalSupplier.LabelUnitIdTrimIndicator, typeof(bool));
			dtLocalSupplier.Columns.Add(TABLE.LocalSupplier.FacilityPrefix, typeof(string));
			dtLocalSupplier.Columns.Add(TABLE.LocalSupplier.PhoneNumber, typeof(string));
			dtLocalSupplier.Columns.Add(TABLE.LocalSupplier.FaxNumber, typeof(string));
			dtLocalSupplier.Columns.Add(TABLE.LocalSupplier.TestingIndicator, typeof(bool));
			dtLocalSupplier.Columns.Add(TABLE.LocalSupplier.CollectionIndicator, typeof(bool));
			dtLocalSupplier.Columns.Add(TABLE.LocalSupplier.RecordStatusCode, typeof(string));
			dtLocalSupplier.Columns.Add(TABLE.LocalSupplier.IccbbaRegistrationNumber, typeof(string));
			dtLocalSupplier.Columns.Add(TABLE.LocalSupplier.LastUpdateFunctionId, typeof(int));

			if (isUpdate)
			{
				dtLocalSupplier.Columns.Add(TABLE.LocalSupplier.RowVersion, typeof(byte[]));
			}
			else
			{
				dtLocalSupplier.Columns.Add(TABLE.LocalSupplier.FacilityGuid, typeof(System.Guid));
				dtLocalSupplier.Columns.Add(TABLE.LocalSupplier.AddressGuid, typeof(System.Guid));
				dtLocalSupplier.Columns.Add(TABLE.LocalSupplier.FdaRegistrationNumber, typeof(string));
				dtLocalSupplier.Columns.Add(TABLE.LocalSupplier.DivisionCode, typeof(string));
			}

			return(dtLocalSupplier);
		}
	}
}